/*
 * File: Snowfall.java
 * ================================================================
 * A program that simulates falling snow on a scene.
 */
import acm.program.*;
import acm.graphics.*;
import acm.util.*;
import java.awt.*;
import java.awt.event.*;

public class Snowfall extends GraphicsProgram {
	/* Preferred window size. */
	public static final int APPLICATION_WIDTH = 500;
	
	/* The delay between frames. */
	private static final double PAUSE_TIME = 1000.0 / 100;
	
	/* How fast the snowflakes fall. */
	private static final double Y_VELOCITY = 5;
	
	/* The size of a snowflake. */
	private static final double SNOWFLAKE_SIZE = 20;
	
	/* The dimensions of the boundaries we place. */
	private static final double BOX_WIDTH = 100;
	private static final double BOX_HEIGHT = 20;
	
	/**
	 * Sets up the mouse listeners.
	 */
	public void init() {
		addMouseListeners();
	}
	
	/**
	 * Drops snowflakes for all to see!
	 */
	public void run() {
		/* Continuously drop a single snowflake. */
		while (true) {
			/* Choose where the snowflake should go. */
			double x = getRandomX();
			GOval snowflake = createSnowflake(x);
			
			/* Add the snowflake.  We send it to the back of the
			 * collage so that later calls to getElementAt don't
			 * pick it up.
			 */
			add(snowflake);
			snowflake.sendToBack();
			
			dropSnowflake(snowflake);
		}
	}
	
	/**
	 * Chooses a random x coordinate for the snowflake.
	 * 
	 * @return A random x coordinate for the snowflake.
	 */
	private double getRandomX() {
		/* Valid x coordinates range from 0 to the width of the
		 * screen minus the snowflake size, so that the upper-left
		 * corner of snowflakes put here don't overflow the screen.
		 */
		RandomGenerator rgen = RandomGenerator.getInstance();
		return rgen.nextDouble(0, getWidth() - SNOWFLAKE_SIZE);
	}
	
	/**
	 * Creates a snowflake at the top of the screen with the
	 * specified x coordinate.
	 * 
	 * @param xCoord The x coordinate of the snowflake.
	 * @return The generated snowflake.
	 */
	private GOval createSnowflake(double xCoord) {
		GOval result = new GOval(xCoord, 0, SNOWFLAKE_SIZE, SNOWFLAKE_SIZE);
		result.setFilled(true);
		result.setColor(Color.GRAY);
		return result;
	}
	
	/**
	 * Simulates the given snowflake dropping
	 * 
	 * @param snowflake The snowflake to drop.
	 */
	private void dropSnowflake(GOval snowflake) {
		/* Loop until the snowflake hits something. */
		while (!isSnowflakeOnGround(snowflake) && !hasSnowflakeCollided(snowflake)) {
			snowflake.move(0, Y_VELOCITY);
			pause(PAUSE_TIME);
		}
	}
	
	/**
	 * Returns whether the given snowflake has hit the ground and
	 * should stop moving.
	 * 
	 * @param snowflake The snowflake in question.
	 * @return Whether it has hit the ground.
	 */
	private boolean isSnowflakeOnGround(GOval snowflake) {
		return snowflake.getY() + snowflake.getHeight() >= getHeight();
	}
	
	/**
	 * Returns whether the given snowflake has hit an object in
	 * the world that would cause it to stop.
	 * 
	 * @param snowflake The snowflake in question.
	 * @return Whether it has hit something else.
	 */
	private boolean hasSnowflakeCollided(GOval snowflake) {
		/* Decide where to check for collisions. */
		double probeX = snowflake.getX() + snowflake.getWidth() / 2.0;
		double probeY = snowflake.getY() + snowflake.getHeight();
		
		/* Look up the object there. */
		GObject collider = getElementAt(probeX, probeY);
		
		/* Return whether we hit something other than the snowflake
		 * itself.
		 */
		return collider != null && collider != snowflake;
	}
	
	/**
	 * Places a barrier at the given click location.
	 */
	public void mouseClicked(MouseEvent e) {
		GRect barrier = new GRect(e.getX() - BOX_WIDTH / 2.0,
				                  e.getY() - BOX_HEIGHT / 2.0,
				                  BOX_WIDTH, BOX_HEIGHT);
		barrier.setFilled(true);
		add(barrier);
	}
}
